﻿using System.Collections.Generic;

namespace Eliza
{
    /// <summary>
    /// Simulates a psychiatrist.
    /// </summary>
    class Eliza : IEliza
    {
        #region Constructor

            /// <summary>
            /// Initializes a new instance of the <see cref="Eliza"/> class.
            /// </summary>
            public Eliza()
            {
                // Initialize Eliza so that it can be used
                this.Initialize();
            }

        #endregion

        #region Public properties

            /// <summary>
            /// Gets the message used to welcome a user to Eliza.
            /// </summary>
            public string WelcomeMessage {
                get { return "WELCOME!  PLEASE TELL ME YOUR PROBLEM."; }
            }

        #endregion

        #region Public methods

            /// <summary>
            /// Gets the response to a user's input.
            /// </summary>
            /// <param name="input">The user's input.</param>
            /// <param name="conversationHasEnded">Flag: the conversation has ended.</param>
            /// <returns>Eliza's response.</returns>
            public string GetResponse
                (string input,
                 out bool conversationHasEnded)
            {
                // Make input uppercase so we don't have to handle mixed-case text
                input = input.ToUpper();

                // Replace multiple spaces with a single space to make it easier for
                // a pattern to search for a phrase
                while (input.Contains("  ")) {
                    input = input.Replace("  ", " ");
                }

                // Remove punctuation marks so we don't have to deal with them
                input = input.Replace(".", string.Empty)
                             .Replace(",", string.Empty)
                             .Replace(";", string.Empty)
                             .Replace("?", string.Empty)
                             .Replace("!", string.Empty);

                // Return "good bye" if the user ended the conversation
                conversationHasEnded = Eliza.DoesInputImplyEndOfConversation(input);
                if (conversationHasEnded) {
                    return Eliza.GetGoodByeResponse();
                }

                // Find matching pattern - if none, return a generic response
                Pattern pattern = FindMatchingPatternForInput (input);
                if (pattern == null) {
                    return GetGenericResponse();
                }

                // Return the pattern's response
                return pattern.GenerateResponse();
            }

        #endregion

        #region Private methods

            /// <summary>
            /// Initializes this instance.
            /// </summary>
            private void Initialize()
            {
                // Initialize the list of patterns that Eliza understands
                this._patterns = new List<Pattern>();
                this._patterns.Add(new Pattern("BECAUSE"));
                this._patterns.Add(new Pattern("YES"));
                this._patterns.Add(new Pattern("NO"));
            }

            /// <summary>
            /// Checks if the input implies the end of the conversation.
            /// </summary>
            /// <param name="input">The input.</param>
            /// <returns>
            /// <see langword="true"/> if the input implies the end of the conversation, otherwise
            /// <see langword="false"/>.
            /// </returns>
            private static bool DoesInputImplyEndOfConversation
                (string input)
            {
                return (input.ToUpper() == "BYE") ||
                       (input.ToUpper() == "GOODBYE") ||
                       (input.ToUpper() == "QUIT");
            }

            /// <summary>
            /// Gets the goodbye response.
            /// </summary>
            /// <returns>
            /// The goodbye response.
            /// </returns>
            private static string GetGoodByeResponse()
            {
                return Eliza.GoodbyeMessage;
            }

            /// <summary>
            /// Gets a generic response.
            /// </summary>
            /// <returns>
            /// A generic response.
            /// </returns>
            private static string GetGenericResponse()
            {
                return Eliza.GenericResponse;
            }

            /// <summary>
            /// Finds the matching pattern for the user's input.
            /// </summary>
            /// <param name="input">The input.</param>
            /// <returns>The pattern or <see langword="null"/> if none was found.</returns>
            private Pattern FindMatchingPatternForInput
                (string input)
            {
                // Check if the input matches a pattern.  If so, return that pattern.
                foreach(Pattern p in this._patterns) {
                    if (p.MatchesInput(input)) {
                        return p;
                    }
                }

                // No pattern was matched, so return null
                return null;
            }

        #endregion

        #region Fields

            /// <summary>
            /// The patterns that Eliza understands.
            /// </summary>
            private List<Pattern> _patterns;

            /// <summary>
            /// The text spoken by Eliza when the conversation has ended.
            /// </summary>
            private const string GoodbyeMessage = "GOODBYE!  IT WAS NICE TALKING WITH YOU.";

            /// <summary>
            /// A generic response by Eliza.
            /// </summary>
            private const string GenericResponse = "PLEASE GO ON.";

        #endregion
    }
}
